//  Alexander Griekspoor (mek@mekentosj.com)
//  Charles Parnot (charles.parnot@stanford.edu)
//  mekentosj.com - Sat Jan 08 2005


// =======================================
// SPRING OBJECT
// =======================================


/*
Usage:

	//create a spring
	var frequency = 2; // 2 oscillations per second
	var damping = 0.1;  // this is quite small damping
	var immobility_threshold = 0.01;
	var initial_position = 0;
	var myspring = new Spring ( frequency, damping, immobility_threshold, initial_position);

	//run the spring
	myspring.setTarget ( 100 ); //always use the accessor
	myspring.move ( 0.01 ); //move the spring for 0.1 seconds
	newposition = myspring.position;
	
	//change the target like somebody suddenly pulled the spring
	myspring.setTarget ( 50 ); //always use the accessor to ensure smooth transition
	myspring.move ( 0.01 ); //move the spring for 0.1 seconds, using the previous speed and position
	newposition = myspring.position;

*/


/*
Dr. J. B. Tatum, thanks very much for this:
http://orca.phys.uvic.ca/~tatum/classmechs/class11.pdf

This spring object implementation makes heavy use of this text.

Of course, this is very classical and other references exist on the web.
But it is nice to not have to redo everything and instead use a clear,
well-written, correct and useful textbook for free.

*/



//some prefedined springs that can be used

//these are 'bouncy' springs
function slow_bouncing_spring () { return new Spring ( 1,   0.25,    0.0001); }
function fast_bouncing_spring () { return new Spring ( 8,   2,    0.0001); }
function shaky_spring ()         { return new Spring ( 50,    12, 0.0001); }

//these are critically damped springs that will get to the target w/o bouncing and w/o wasting time
function fast_spring ()          { return new Spring ( 8,   16,    0.0001); }
function very_fast_spring ()     { return new Spring ( 15,   30,    0.0001); }
function super_fast_spring ()    { return new Spring ( 30,   60,    0.0001); }
function slow_spring ()          { return new Spring ( 2,   4,    0.0003); }
function very_slow_spring ()     { return new Spring ( 0.1,   0.2,    0.001); }

//this one is highly dampened
function super_slow_spring ()    { return new Spring ( 0.5,   4,    0.0001); }




// frequency = oscillation frequency in the absence of damping
// damping = damping factor, from friction force

//constructor
function Spring ( frequency, damping, immobility ) {

	if ( frequency <= 0 )
		frequency = 0.001;
		
	//basics
	this.frequency = frequency;
	this.damping = damping;
	this.immobility_threshold = immobility;

	//the spring starts immobile at position 0, time 0
	this.time = 0;
	this.speed = 0;
	this.position = 0;
	this.target = 0;
	this.is_moving = false;

	//methods
	this.setTarget = setTarget;
	this.setFrequency = setFrequency;
	this.reset_damping_type = reset_damping_type;
	this.move = move_spring_private;

	//what type of damping do we have?
	//this will determine the equation used to calculate movements
	this.reset_damping_type();
	this.reset_initial_conditions();
}

function reset_damping_type ()
{
	/*
	//special case: no damping = spring always moving
	if ( this.damping == 0 ) {
		this.update_position = update_position_no_damping;
		this.reset_initial_conditions = reset_initial_conditions_no_damping;
		return;
	}
	*/
	
	//what type of damping have we: light or heavy?
	//this will determine the equation used to calculate movements
	this.d = this.frequency * this.frequency - 0.25 * this.damping * this.damping;
	if ( this.damping < 2 * this.frequency ) {
		this.update_position = update_position_light_damping;
		this.reset_initial_conditions = reset_initial_conditions_light_damping;
		this.w = Math.sqrt ( this.d );		
	} else if ( this.damping > 2 * this.frequency ) {
		this.update_position = update_position_heavy_damping;
		this.reset_initial_conditions = reset_initial_conditions_heavy_damping;
		this.w = Math.sqrt ( - this.d );		
		this.l1 = this.damping / 2 - this.w;
		this.l2 = this.damping / 2 + this.w;
	} else {
		this.update_position = update_position_critical_damping;
		this.reset_initial_conditions = reset_initial_conditions_critical_damping;
	}
	
}

/*
//the equations of movement for no damping are:
function reset_initial_conditions_no_damping ()
{
	
}
*/

//the equations of movement for light damping are:
//    x  = C exp ( - damping t / 2 ) sin ( w t + alpha )
//    x' = C exp ( - damping t / 2 ) ( w cos ( wt + alpha ) - damping sin ( w t + alpha ) / 2 )  
function reset_initial_conditions_light_damping ()
{
	//determine new values for alpha (=this.al) and constant C (=this.c)
	delta = this.position - this.target;
	if ( delta == 0 ) {
		if (this.speed > 0)
			this.al = + Math.PI;
		else
			this.al = - Math.PI;
		this.c = this.speed / this.frequency * Math.cos(this.al);
	} else {
		cotal = ( this.speed / delta + this.damping / 2 ) / this.w;
		if ( cotal == 0 ) {
			if  ( delta > 0 )
				this.al = + Math.PI / 2;
			else
				this.al = - Math.PI / 2;
			this.c = delta;
		} else {
			this.al = Math.atan (1/cotal);
			//note that this.a can't be 0 so we are fine
			this.c = delta / Math.sin(this.al);
			//if necessary, correct the angle to get the right quadrant and have this.c > 0
			//note that the case this.c = 0 does not have to be covered, as we have already treated delta = 0
			if  ( this.c < 0 ) {
				this.al += Math.PI;
				this.c = -this.c;
			}
		}
	}
}

//the equations of movement for heavy damping are:
//    x  = A exp ( - l1 t ) + B exp ( -l2t )
//    x' = - A l1 exp ( - l1 t ) - B l2 exp ( -l2t )
function reset_initial_conditions_heavy_damping ()
{
	//determine new values for constants A and B (=this.a and this.b)
	delta = this.position - this.target;
	this.a = ( this.speed + this.l2 * delta ) / ( 2 * this.w );
	this.b = delta - this.a;
}

//the equations of movement for critical damping are:
//    x  = A exp ( - damping t / 2 ) + B t exp ( - damping t / 2 )
//    x' = ( - damping A / 2 + B ) exp ( - damping t / 2 ) - damping B t exp ( - damping t / 2 ) / 2
function reset_initial_conditions_critical_damping ()
{
	//determine new values for constants A and B (=this.a and this.b)
	//where x = A exp(-gamma/2 t) + B t exp(-gamma/2 t)
	delta = this.position - this.target;
	this.a = delta;
	this.b = this.speed + this.damping * delta / 2;
}


function setTarget ( target )
{
	this.target = target;
	this.time = 0;
	this.reset_initial_conditions();
}

function setFrequency ( frequency )
{
	this.frequency = frequency;
	this.time = 0;
	this.reset_damping_type();
	this.reset_initial_conditions();
}


function move_spring_private ( dt )
{
	//test for immobility, which happens when position is close enough to target AND the speed is close to zero
	var scale;
	if ( this.target == 0 )
		scale = 100; //hackish; I know it will happen only with the counter spring, so I break the encapsulation of the view and the model and stop pretending the model does not know what it is used for; argh
	else
		scale = Math.abs(this.target) + Math.abs(this.position);// + Math.MIN_VALUE * 100;
	var delta = this.target - this.position;
	var norm_speed = Math.abs ( this.speed * dt / scale );
	var norm_position = Math.abs ( delta / scale );
	if (  norm_speed < this.immobility_threshold &&  norm_position < this.immobility_threshold ) {
		this.position = this.target;
		this.speed = 0;
		this.is_moving = false;
	}
	
	//not immobile... hence moving!
	else {
		this.is_moving = true;
		this.time += dt;
		this.update_position();
		if ( this.position < 0 ) {
			//debug ( "negative value!!");
			this.position = -this.position;
			this.speed = -this.speed;
			this.time = 0;
			this.reset_initial_conditions ();
		}
	}
}

//the equations of movement for light damping are:
//    x  = C exp ( - damping t / 2 ) sin ( w t + alpha )
//    x' = C exp ( - damping t / 2 ) ( w cos ( wt + alpha ) - damping sin ( w t + alpha ) / 2 )  
function update_position_light_damping ( )
{
	//debug ("update_position_light_damping");
	exptime = Math.exp ( - this.damping * this.time / 2 );
	wtime = this.w * this.time + this.al;
	this.position = this.target + this.c * exptime * Math.sin ( wtime );
	this.speed = this.c * exptime * ( this.w * Math.cos(wtime) - this.damping * Math.sin(wtime) / 2 );
}

//the equations of movement for heavy damping are:
//    x  = A exp ( - l1 t ) + B exp ( -l2t )
//    x' = - A l1 exp ( - l1 t ) - B l2 exp ( -l2t )
function update_position_heavy_damping ( )
{
	//debug ("update_position_heavy_damping");
	exp1 = Math.exp ( - this.l1 * this.time );
	exp2 = Math.exp ( - this.l2 * this.time );
	this.position = this.target + this.a * exp1 + this.b * exp2;
	this.speed = - this.a * this.l1 * exp1 - this.b * this.l2 * exp2;
}

//the equations of movement for critical damping are:
//    x  = A exp ( - damping t / 2 ) + B t exp ( - damping t / 2 )
//    x' = ( - damping A / 2 + B ) exp ( - damping t / 2 ) - damping B t exp ( - damping t / 2 ) / 2
function update_position_critical_damping ( )
{
	//debug ("update_position_critical_damping");
	exptime = Math.exp ( - this.damping * this.time / 2 );
	this.position = this.target + this.a * exptime + this.b * this.time * exptime;
	this.speed = ( - this.damping * this.a / 2 + this.b ) * exptime - this.damping * this.b * this.time * exptime / 2;
}


