//  Alexander Griekspoor (mek@mekentosj.com)
//  Charles Parnot (charles.parnot@stanford.edu)
//  mekentosj.com - Sat Jan 08 2005


// =======================================
// SKIN2 =  'CLASSIC'
// =======================================


//Skin2 object constructor
//all the functions are defined INSIDE that function, to isolate the namespace inside it
//all methods are thus in fact closures, and every instance of Skin1 get a separate function object for each of the methods but that's OK because skins are in singleton anyway so only one instance is created

//this code was setup this way so it is easy to copy and paste it to create a new skin without overlapping with the global namespace


function Skin2()
{

//debug ("creating an instance of Skin2");

// =======================================
// GENERAL SETUP
// =======================================

//all these different parts are animated independently
this.tachometer = new Animation ( reset_tachometer, move_tachometer, draw_tachometer, 30 );
this.counter = new Animation ( reset_counter, move_counter, draw_counter, 40 );
this.light = new Animation ( reset_light, move_light_during_transition, draw_light, 10 );

//some of the Animation objects use a spring to model their behavior
this.tachometer.spring = slow_spring();
this.counter.spring = shaky_spring();

//the status light makes use of two objects
// - pulse  = infinite movement, no damping = pulsing light
// - spring = for transition = go from one color to another
this.light.pulse = new Pulse ( 0.1, 60, 80 );
this.light.transition = slow_spring();
this.light.status = 0;
this.light.transition_initial = 0;
this.light.transition_final = 0;
this.light.start_transition = start_transition;

//methods that all skins should respond to
this.sync = sync;
this.start = start;
this.stop = stop;
this.click = click;

//keys to use to retrieve the number to be displayed in the counter
this.keys = {
	agent: [
		'workingAgentCount',
		'availableAgentCount',
		'unavailableAgentCount',
		'offlineAgentCount',
		'totalAgentCount',
		'workingAgentPercentage'
	],
	processor: [
		'workingProcessorCount',
		'availableProcessorCount',
		'unavailableProcessorCount',
		'offlineProcessorCount',
		'onlineProcessorCount',
		'workingAgentPercentage'
	]
};

//depending on the type of info displayed, the counter may use a decimal or not
this.decimals = [false,false,false,false,false,true];

//for debugging
this.tachometer.name="tachometer";
this.counter.name="counter";
this.light.name="light";


// =======================================
// DRAWING PARAMETERS
// =======================================

//the following variables could have been properties of the skin instead, but that was just easier this way
//whatever... the usage is the same in the end, as everything is inside the constructor and methods are in fact closures

//dimensions for the light
var light_w = 12;
var light_h = 12;
var light_x = 156;
var light_y = 85;

//dimensions for the counter legend
var legend_w = 41;
var legend_h = 10;
var legend_x = 113;
var legend_y = 85;


//this array will be used to automatically create image objects and declare variables for them (see below)
var images = {
	base:{   path:'skin2/base.png', x:0 , y:0 , w:200 , h:200 },
	glow:{  path:'skin1/glow.png', x:10 , y:10 , w:180 , h:100 },

	basecut:{   path:'skin2/base-cut.png', x:0 , y:0 , w:200 , h:200 },
	needle:{   path:'skin1/needle.png', x:0 , y:0 , w:8 , h:70 },

	mhz:{  path:'skin1/mhz.png',  x:78 , y:122 , w:45 , h:20 },
	ghz:{  path:'skin1/ghz.png',  x:78 , y:122 , w:45 , h:20 },
	thz:{  path:'skin1/thz.png',  x:78 , y:122 , w:45 , h:20 },
	x10:{  path:'skin1/x10.png',  x:82 , y:142 , w:35 , h:15 },
	x100:{ path:'skin1/x100.png', x:82 , y:142 , w:35 , h:15 },
	x1000:{ path:'skin1/x1000.png', x:82 , y:142 , w:35 , h:15 },
	
	counter_base:{  path:'skin2/counter_base.png', x:113 , y:96 , w:42 , h:14 },
	counter_mask:{  path:'skin1/counter_mask.png', x:113 , y:96 , w:42 , h:14 },
	decimals:{   path:'skin1/decimals.png', x:0 , y:0 , w:8 , h:150 },
	numbers:{   path:'skin1/numbers.png', x:0 , y:0 , w:8 , h:150 },
	
	working:{  path:'skin2/working.png',         x:legend_x , y:legend_y , w:legend_w , h:legend_h },
	available:{  path:'skin2/available.png',     x:legend_x , y:legend_y , w:legend_w , h:legend_h },
	unavailable:{  path:'skin2/unavailable.png', x:legend_x , y:legend_y , w:legend_w , h:legend_h },
	total:{  path:'skin2/total.png',             x:legend_x , y:legend_y , w:legend_w , h:legend_h },
	offline:{  path:'skin2/offline.png',         x:legend_x , y:legend_y , w:legend_w , h:legend_h },
	percentage:{  path:'skin2/percentage.png',   x:legend_x , y:legend_y , w:legend_w , h:legend_h },

	light_bevel:{   path:'skin1/light-bevel.png', x:light_x , y:light_y , w:light_w , h:light_h },
	light_black:{   path:'skin1/light-black.png', x:light_x , y:light_y , w:light_w , h:light_h },
	light_orange:{   path:'skin1/light-orange.png', x:light_x , y:light_y , w:light_w , h:light_h },
	light_green:{   path:'skin1/light-green.png', x:light_x , y:light_y , w:light_w , h:light_h },
	light_red:{   path:'skin1/light-red.png', x:light_x , y:light_y , w:light_w , h:light_h }
};
	
//load the images for skin 1 by creating images and declaring variables
for ( name in images ) {
	//debug ("creating ");
	var one_image = new Image (images[name].w,images[name].h);
	one_image.src = images[name].path;
	images[name].image = one_image;
	//create variable with the name of the image, so that we can refer to images using simple variable names
	eval ( 'var ' + name + ' = images[name]; ' );
}

//legend index to use for the counter
this.counter.legend_index = 0;
this.counter.legend_images = [ working,available, unavailable,offline,total,percentage];


//used to pick the right image for the light, depending on the status
var status_images = [ light_red,light_orange,light_green, light_red];

//angles at which the needle of the tachometer should be drawn
var beginangle = 0.85; 						// angle for 0
var endangle = 5.45;						// angle for 8
var deltaangle = endangle - beginangle;		// dynamic range

//for each scale used to display the speed, there is a separate set of images to be used
var speed_scales = [
	{ min:0, max:8, images:[ basecut,mhz ]},
	{ min:0, max:80, images:[ basecut,mhz,x10 ]},
	{ min:0, max:800, images:[ basecut,mhz,x100 ]},
	
	{ min:0, max:8000, images:[ basecut,ghz ]},
	{ min:0, max:80000, images:[ basecut,ghz,x10 ]},
	{ min:0, max:800000, images:[ basecut,ghz,x100 ]},
	{ min:0, max:8000000, images:[ basecut, ghz,x1000 ] },

	{ min:0, max:80000000, images:[ basecut, thz,x10 ] },
	{ min:0, max:800000000, images:[ basecut, thz,x100 ] },
];




// =======================================
// SKIN METHODS (SEE PROTOCOL IN MAIN JS)
// =======================================

//this is called by the main script when the values to be displayed have been changed
function sync ()
{

	//reset values for the counter
	this.counter.reset();
	var new_target = agents[this.keys[display_pref][this.counter.legend_index]];
	if ( ( new_target == undefined ) || isNaN(new_target) )
		new_target = 0;
	this.counter.spring.setTarget(new_target);
	this.counter.decimal = this.decimals[this.counter.legend_index];
	
	//reset tachometer
	this.tachometer.reset();
	this.tachometer.spring.setTarget(speed);
	
	//reset status light
	this.light.reset();
	this.light.start_transition();
	
};


//start skin animation
function start ()
{
	//return;
	
	this.tachometer.start();
	
	//read section below about overlay mode and switching drawing modes
	//in overlay mode, only the tachometer animation is in control
	if ( this.tachometer.overlay == false ) {
		this.counter.start();
		this.light.start();
	}
}

//stop skin animation
function stop ()
{
	this.tachometer.stop();
	this.counter.stop();
	this.light.stop();
}

//clicking should change:
//  - the legend for the counter
//  - the value displayed by the counter
function click ()
{
	this.counter.legend_index ++;
	if (this.counter.legend_index>5)
		this.counter.legend_index = 0;
	
	//update the drawing
	this.sync();
	this.start();
}


// =======================================
// SWITCHING DRAWING MODE
// =======================================

//normally, the different elements (tachometer, counter and light) are drawn separately, each controlled by a different animation object

//however, when the speed is between the critical values where the tachometer needle will overlay the counter and status light,
//we need to switch to a different drawing mode, where everything is drawn at the same time and all at the same fast framerate

//for this, we need to call a method on the skin itself, which will be called by the move method of the tachometer object
//thus, the tachometer needs to keep a pointer to the parent skin
this.tachometer.skin = this;

//the tachometer will use that flag to know what to draw when the draw method is called
//if the switch is on, then the counter and the light will also be drawn
this.tachometer.overlay = true;

//and here is the switch method that will be called on the skin when appropriate
this.switchDrawingMode = switchDrawingMode;
function switchDrawingMode ()
{
	var displayed_speed = this.tachometer.spring.position;
	var displayed_scale_index = scale_index(displayed_speed);
	var displayed_needle_position = displayed_speed / speed_scales[displayed_scale_index].max;
	if (displayed_needle_position > 5/8) {
		if (!this.tachometer.overlay) {
			//debug("Enabling overlay mode at speed = "+displayed_speed );
			this.tachometer.overlay = true;
			this.tachometer.framerate = 40;
			this.counter.stop();
			this.light.stop();
		}
	} else if (this.tachometer.overlay) {
		//debug("Disabling overlay mode at speed = "+displayed_speed );
		this.tachometer.overlay = false;
		this.tachometer.framerate = 30;
		this.counter.start();
		this.light.start();
	}
}



// =======================================
// DRAWING THE TACHOMETER
// =======================================

//drawing the tachometer involves several elements:
// - the right numbers for the right scale
// - the right labeling for the unit (usually GHz, but MHz can still happen if only one machine; and TeraHz, well,...)
// - the needle

//last values used to draw the different components
//these are used to avoid useless drawing (e.g. if position of the element did not change)
function reset_tachometer ( )
{
	this.scale = -1;
	this.angle = -1000;
}

//moving the tachometer is about moving the spring used to model it
//this method is also used by the disk and the counter
function move_tachometer ( dt )
{
	//here we check wether the speed is between the critical values where the tachometer needle will overlay the counter and status light
	//when the overlay occurs, we need to switch to a different drawing mode, where everything is drawn at the same time and all at the same fast framerate
	//for this, we need to call a method on the skin itself
	this.skin.switchDrawingMode();

	//when in overlay mode, there is only one animation running, controlled by the tachometer animation
	if ( this.overlay ) {
		this.spring.move(dt);
		this.skin.counter.move(dt);
		this.skin.light.move(dt);

	//moving the light ?
	var light_needs_animation = true;
	if ( this.skin.light.transition.is_moving == false && (
				(agent_status == status_off && this.skin.light.status == status_off ) ||
				(agent_status == status_disconnected && this.skin.light.status == status_disconnected) ) )
		light_needs_animation = false;

	//adjust framerate
	var new_framerate = 0;
	if ( light_needs_animation )
		new_framerate = 10;
	if ( this.spring.is_moving )
		new_framerate = 30;
	if ( this.skin.counter.spring.is_moving )
		new_framerate = 40;
	
	//stop only if no animation at all
	if ( new_framerate == 0 )
		this.stop();
	else
		this.framerate = new_framerate;

	}
	
	//otherwise, only the tachometer needs to move
	else {
		this.spring.move(dt);
		if ( this.spring.is_moving == false )
			this.stop();
	}

}


//this function is used to return the index in the speed_scales that correspond to speed_value
function scale_index ( speed_value ) {
	var n = speed_scales.length;
	for ( i=0; i<n; i++ ) {
		var max = speed_scales[i].max;
		if ( speed_value < max )
			break;
	}
	return i;
}


function draw_tachometer ( )
{
	//debug ("draw_tacho start");
	// get canvas and context
	var canvas  = document.getElementById("canvas");	
	var context = canvas.getContext("2d");
	
	//determine the scale to be used
	// if current_speed > target_speed, then use a scale appropriate for current_speed
	// if current_speed < target_speed, then use a scale appropriate for target_speed except if the spring is slow enough
	var scale_index_current = scale_index ( this.spring.position );
	var scale_index_target = scale_index ( this.spring.target );
	var scale_index_now;
	if ( scale_index_current >= scale_index_target )
		scale_index_now = scale_index_current;
	else {
		scale_index_now = scale_index_target;
		if ( this.spring.w < 0.1 ) //oscillation of the spring takes more than 10 seconds?
			scale_index_now --;
	}
	
	//new angle of the meter
	meterangle =  beginangle + ( deltaangle * (this.spring.position / speed_scales[scale_index_now].max ) );

	//if the scale just changed, redraw everything
	if ( this.scale != scale_index_now ) {
		// clear context
		//context.clearRect (0, 0, 200, 200);
		//draw the scale elements as stored in the speed_scales array
		var elements = speed_scales[scale_index_now].images;
		for ( i = 0; i < elements.length ; i++ )
			context.drawImage (elements[i].image, elements[i].x, elements[i].y, elements[i].w, elements[i].h );
		//add the glow
		context.drawImage (glow.image, glow.x, glow.y, glow.w, glow.h );
		//remember last scale used for next time
		this.scale = scale_index_now;
	}


	//before drawing the needle, we need to clear the old and the new positions of the needle to remove the old needle and then to be able to draw the glow on top of it
	//the rectangle to clear is a 'straight' rectangle that has to enclose the two positions of the needle, that are rotated rectangles
	//for this, we use a few functions from 'utilities.js' that deal with rectangles
	
	//rectangle for the new needle
	var needle_rect = {x:-5,y:6,w:10,h:72};
	var clear_rect = straight_rectangle ( needle_rect , meterangle );

	//union it with the rectangle for the old position of the needle
	var old_rect = straight_rectangle ( needle_rect , this.angle );
	clear_rect = union (old_rect, clear_rect);
	this.angle = meterangle;

	//note the translation to (100,100), which was the center of the rotation for the calls to the 'straight_rectangle' function
	clear_rect.x += 100;
	clear_rect.y += 100;
	
	//redraw the scale elements in the rectangle to clear
	var elements = speed_scales[scale_index_now].images;
	var redraw_rect;
	for ( i = 0; i < elements.length ; i++ ) {
		redraw_rect = intersection ( clear_rect, elements[i] );
		//debug ( "intersection " + string_from_rect(redraw_rect) );
		if ( redraw_rect.w != 0 && redraw_rect.h != 0 )
			context.drawImage (elements[i].image, redraw_rect.x-elements[i].x, redraw_rect.y-elements[i].y, redraw_rect.w, redraw_rect.h, redraw_rect.x, redraw_rect.y, redraw_rect.w, redraw_rect.h);
	}
	
	//in overlay drawing mode, also draw the counter and the light
	if ( this.overlay ) {
		this.skin.counter.draw();
		this.skin.light.draw();
	}
	
	//draw the needle
	context.save();
		context.translate (100,100);
		context.rotate (meterangle);
		if ( grid_status == status_off )
			context.globalAlpha = 0.5;
		context.drawImage (needle.image, -4, 7, needle.w, needle.h);
	context.restore();

	//redraw the glow on the portion that was cleared
	redraw_rect = intersection ( clear_rect, glow );
	if ( redraw_rect.w != 0 && redraw_rect.h != 0 ) 
		context.drawImage (glow.image, redraw_rect.x-glow.x, redraw_rect.y-glow.y, redraw_rect.w, redraw_rect.h, redraw_rect.x, redraw_rect.y, redraw_rect.w, redraw_rect.h);

	//debug ("draw_tacho end");
}


// =======================================
// DRAWING THE COUNTER
// =======================================


/* the difference between the counter in Skin1 and Skin2 is that
we redraw everything all the time, regardless of previous drawing.
This is because the needle might be in the way.
Also, in skin2, the legend of the counter is drawn at the same time
as the counter.*/

//display of the counter
var begin_counter = 0;                          		// offset for 0
var end_counter = 136;                          		// offset for 9
var span_counter = end_counter - begin_counter;  		// dynamic range

function reset_counter ( )
{
	/*no code here, unlike Skin1
	//reset last values used to draw the different components
	//these are used to avoid useless drawing (e.g. if position of the element did not change)
	this.digits = [ -1, -1, -1, -1 ];
	*/
}

function move_counter ( dt )
{
	this.spring.move(dt);
	if ( this.spring.is_moving == false )
		this.stop();
}


function draw_counter ()
{
	//debug ("start draw_counter");

	// get canvas and context
	var canvas  = document.getElementById("canvas");	
	var context = canvas.getContext("2d");
	
	//draw the legend for the counter: 'Working', 'Available',...
	var legend = this.legend_images[this.legend_index];
	context.clearRect(legend.x, legend.y, legend.w, legend.h);
	context.drawImage (legend.image, legend.x, legend.y, legend.w, legend.h);

	
	//digit offset in pixels
	//digit at index 0 = right
	var digits = [ -2, -2, -2, -2 ];

	//get the offset for the first digit, which is special in 2 ways:
	// - it can be a decimal
	// - its position is fractional in all cases
	var nr; //digit value to be displayed
	var c; //value to be displayed
	if ( this.decimal == true )
		c = this.spring.position * 10;
	else
		c = this.spring.position;
	nr = c % 10;
	digits[0] = Math.round ( begin_counter + nr * (span_counter / 10.0) );
	
	
	//if the first digit is close to 10, then we need to start moving some of the other digits
	//(those adjacent and also above 9, see below)
	var should_add_fraction = false;
	var fraction_to_add;
	if ( nr > 9 ) {
		should_add_fraction = true;
		var a = nr - 9.0;
		//this is what will be added to digits about to change (use the square for smoother movement)
		fraction_to_add = a * a;
	}

	//now loop through the remaining digits to determine their position
	for ( i=1; i<4; i++ ) {
		c = Math.floor (c/10);
		var nr = c % 10;
		if ( nr < 0 )
			nr += 10;
		//add a fractional part if necessary
		if ( should_add_fraction==true ) {
			nr += fraction_to_add;
			//debug ( "add to digit " + i + "fraction " + fraction_to_add + " = " + nr );
			//should we keep adding fraction?
			if ( nr < 9 )
				should_add_fraction = false;
		}
		digits[i] = Math.round ( begin_counter + nr * (span_counter / 10.0) );
	}
	
	//the drawing itself
	var imax = 4;
	digx = counter_mask.x + counter_mask.w - 10;
	digy = counter_mask.y + 2;
	//erase the digits with counter background image
	context.drawImageFromRect(counter_base.image,  digx-imax*10+10-counter_base.x, 0, 10*imax, counter_base.h, digx-imax*10+10, counter_base.y, 10*imax, counter_base.h,"copy");
	//draw the digits
	if ( this.decimal == true )
		context.drawImageFromRect(decimals.image, 0, digits[0], 8, 12, digx, digy, 8, 12,"source-over");
	else
		context.drawImageFromRect(numbers.image,  0, digits[0], 8, 12, digx, digy, 8, 12,"source-over");
	for ( i=1; i<imax;i++ )
		context.drawImageFromRect(numbers.image,  0, digits[i], 8, 12, digx-i*10, digy, 8, 12,"source-over");
	//draw the mask
	context.drawImageFromRect(counter_mask.image,  0, 0, 10*imax, counter_mask.h, digx-imax*10+10, counter_mask.y, 10*imax, counter_mask.h,"source-over");
	
}


// =======================================
// DRAWING THE STATUS LIGHT
// =======================================


this.light.draw_light_with_status = draw_light_with_status;

function reset_light ( )
{
	//reset last values used to draw the different components
	//these are used to avoid useless drawing (e.g. if position of the element did not change)
	this.start_transition();
}

function move_light (dt)
{
	this.pulse.move(dt);
	if ( (agent_status == status_off && this.status == status_off ) ||
			(agent_status == status_disconnected && this.status == status_disconnected) )
		this.stop();
}

function move_light_during_transition ( dt )
{
	//move the pulse
	this.pulse.move(dt);
	//move the spring controlling the transition
	this.transition.move(dt);
	if ( this.transition.is_moving == false ) {
		//debug ("end transition");
		this.move = move_light;
		this.status = this.transition_final;
		//maybe a new transition is necessary?
		this.start_transition();
	}
}

function start_transition ()
{
	//only start a transition if the previous is done and there is a transition to do
	if ( this.transition.is_moving == false  && this.status != agent_status ) {
		//debug ( "starting transition from " + this.status + " to " + agent_status);
		this.transition_initial = this.status;
		this.transition_final = agent_status;
		this.transition = slow_spring();
		this.transition.setTarget(1.0);
		this.transition.is_moving = true;
		this.status = agent_status;
		//switch to new mode
		this.move = move_light_during_transition;
		//adjust pulse frequency
		if ( agent_status == status_working )
			this.pulse.setFrequency(0.4);
		else if ( agent_status == status_available )
			this.pulse.setFrequency(0.1);
	}
}


function draw_light_with_status ( base_alpha, status )
{
	var img = status_images[status];
	//debug(base_alpha);
	//debug(status);

	// get canvas and context
	var canvas  = document.getElementById("canvas");	
	var context = canvas.getContext("2d");
	
	//case of a pulsed light
	if ( status == status_working || status == status_available ) {
		var alpha = this.pulse.position  / 100.0;
		//if ( Math.abs(context.globalAlpha-0.5) < 0.1 )
		//	debug("alpha in  = "+ context.globalAlpha);
		context.save();
			context.globalAlpha = base_alpha * alpha;
			context.drawImage(img.image, img.x, img.y, img.w, img.h);
			img = light_black;
			context.globalAlpha = base_alpha * (1.0 - alpha);
			context.drawImage(img.image, img.x, img.y, img.w, img.h);
		context.restore();
		
	}

	//case of a still light
	else {
		context.save();
			context.globalAlpha = base_alpha;
			context.drawImage(img.image, img.x, img.y, img.w, img.h);
		context.restore();
	}
}

function draw_light ()
{
	// get canvas and context
	var canvas  = document.getElementById("canvas");	
	var context = canvas.getContext("2d");
	
	//clear the area with the bevel image
	var img = light_bevel;
	context.save();
		context.globalCompositeOperation = 'copy';
		context.drawImage(img.image, img.x, img.y, img.w, img.h);
	context.restore();
	
	//in case of a transition, we need to overlay 2 images
	if ( this.transition.is_moving ) {
		context.save();
			this.draw_light_with_status(1.0-this.transition.position, this.transition_initial);
			this.draw_light_with_status(this.transition.position, this.transition_final);
		context.restore();
	}
	
	//otherwise, just 1 image
	else
		this.draw_light_with_status(1.0,this.status);
	
	//debug ( "drawLight end" );

}


}
